<?php
/* --------------------------------------------------------------
 ConfigurationReader.php 2019-12-03
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2019 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

declare(strict_types=1);

namespace Gambio\Core\Configuration\Repositories\Components;

use Doctrine\DBAL\Connection;
use Gambio\Core\Configuration\Models\Read\ConfigurationGroupId;
use function array_values;
use function implode;

/**
 * Class ConfigurationReader
 * @package Gambio\Core\Configuration\Repositories\Components
 */
class ConfigurationReader
{
    /**
     * @var Connection
     */
    private $connection;
    
    
    /**
     * ConfigurationReader constructor.
     *
     * @param Connection $queryBuilder
     */
    public function __construct(Connection $queryBuilder)
    {
        $this->connection = $queryBuilder;
    }
    
    
    /**
     * Finds a configuration item by the given key.
     * Returns null if no matching item was found.
     *
     * @param string $key
     *
     * @return array|null
     */
    public function find(string $key): ?array
    {
        $qb = $this->connection->createQueryBuilder();
        
        $select    = $this->createSelect(['key', 'value']);
        $where     = $this->connection->quoteIdentifier('key') . ' = ' . $qb->createNamedParameter($key);
        $langWhere = $qb->expr()->isNull('language_id');
        
        $qb->select($select)->from('gx_configurations')->where($where)->andWhere($langWhere);
        $result = $qb->execute()->fetch();
        
        return $result ? : null;
    }
    
    
    /**
     * Finds a configuration item by the given key and language code.
     * Returns null if no matching item was found.
     *
     * @param string $key
     * @param string $languageCode
     *
     * @return array|null
     */
    public function findWithLanguageCode(string $key, string $languageCode): ?array
    {
        $qb = $this->connection->createQueryBuilder();
        
        $select    = $this->createSelect(['key', 'value']);
        $where     = $this->connection->quoteIdentifier('key') . ' = ' . $qb->createNamedParameter($key);
        $langWhere = $qb->expr()->andX($qb->expr()->eq('l.code', $qb->createNamedParameter($languageCode)));
        
        $qb->select($select)
            ->from('gx_configurations')
            ->where($where)
            ->join('gx_configurations', 'languages', 'l', 'gx_configurations.language_id = l.languages_id')
            ->andWhere($langWhere);
        
        $result = $qb->execute()->fetch();
        
        return $result ? : null;
    }
    
    
    /**
     * Finds a configuration item by the given key and language id.
     * Returns null if no matching item was found.
     *
     * @param string $key
     * @param int    $languageId
     *
     * @return array|null
     */
    public function findWithLanguageId(string $key, int $languageId): ?array
    {
        $qb = $this->connection->createQueryBuilder();
        
        $select    = $this->createSelect(['key', 'value']);
        $where     = $this->connection->quoteIdentifier('key') . ' = ' . $qb->createNamedParameter($key);
        $langWhere = $qb->expr()->andX($qb->expr()->eq('language_id', $qb->createNamedParameter($languageId)));
        
        $qb->select($select)->from('gx_configurations')->where($where)->andWhere($langWhere);
        
        $result = $qb->execute()->fetch();
        
        return $result ? : null;
    }
    
    
    /**
     * Fetches configuration data, filtered by the given group identifier.
     *
     * @param ConfigurationGroupId $group
     * @param array                $excludes
     *
     * @return array
     */
    public function fetchByGroup(ConfigurationGroupId $group, array $excludes = []): array
    {
        // wrapped helper function to create backticks for column definitions
        $bt = function (string $subject) {
            return $this->connection->quoteIdentifier($subject);
        };
        $qb = $this->connection->createQueryBuilder();
        
        $selects = ['key', 'value', 'type', 'l.code'];
        
        $select = $this->createSelect($selects);
        $qb->select($select)
            ->from($bt('gx_configurations'), $bt('c'))
            ->leftJoin($bt('c'),
                       $bt('languages'),
                       $bt('l'),
                       "{$bt('c.language_id')} = {$bt('l.languages_id')}")
            ->where($bt('legacy_group_id') . ' = ' . $qb->createPositionalParameter($group->group()))
            ->orderBy($bt('c.sort_order'))
            ->addOrderBy($bt('c.id'));
        
        foreach ($excludes as $exclude) {
            $qb->andWhere($bt('key') . ' != ' . $qb->createPositionalParameter($exclude));
        }
        
        $result = $qb->execute()->fetchAll();
        
        return $this->mergeLanguageData($result);
    }
    
    
    /**
     * Checks if configuration value for given key is available.
     *
     * @param string $key
     *
     * @return bool
     */
    public function has(string $key): bool
    {
        $qb = $this->connection->createQueryBuilder();
        
        $where = "{$this->connection->quoteIdentifier('key')} = {$qb->createNamedParameter($key)}";
        $qb->select('*')->from('gx_configurations')->where($where);
        
        return $qb->execute()->rowCount() > 0;
    }
    
    
    /**
     * Merges language data into a 'lang' key of the result set.
     *
     * @param array $result
     *
     * @return array
     */
    private function mergeLanguageData(array $result): array
    {
        $data = [];
        foreach ($result as $dataSet) {
            $key   = $dataSet['key'];
            $value = $dataSet['value'];
            
            $data[$key]['key']  = $key;
            $data[$key]['type'] = $dataSet['type'];
            
            // merges language specific data into 'lang', if language code is available
            if ($dataSet['code'] !== null) {
                $data[$key]['lang'][$dataSet['code']] = $value;
                $data[$key]['type']                   = 'lang';
            } else {
                $data[$key]['value'] = $value;
            }
        }
        
        return array_values($data);
    }
    
    
    /**
     * Creates a select query string from the given fields.
     * Backticks are applied to the final returned string.
     *
     * @param array $selectedFields
     *
     * @return string
     */
    private function createSelect(array $selectedFields): string
    {
        $fields = [];
        
        foreach ($selectedFields as $selectedField) {
            $fields[] = $this->connection->quoteIdentifier($selectedField);
        }
        
        return implode(', ', $fields);
    }
}